% This Code performs calculations that replicate data appearing in Table A2 
% of the Online Data Appendix, and data and model simulation results 
% appearing in Tables 2 and 3, and in Figures 1 and 2 of the paper 
% "Saving Rates and Portfolio Choice with Subsistence Consumption" 
% by Carolina Achury, Sylwia Hubar, and Christos Koulovatianos
%
% Nottingham, January 26, 2011
%
% ATTENTION: in order to make this program run properly you must place in 
% the same folder the following dat files that accompany our replication
% zip file:
%
% 1. Mean_Value_of_Financial_Assets.dat
% 2. Family_Net_Worth.dat
% 3. Stock_Holdings_as_Share_of_Financial_Assets.dat
% 4. Before_Tax_Family_Income.dat
% 5. Effective_Marginal_Tax_Rate.dat

clear all;
clc;

% ************************************************************************
% ********* Set Parameter values (Note: these parameter values ***********
% ********* affect the simulated model, but the risk-free rate ***********
% ********* and the return on risky assets affect discount factors *******
% ********* in the construction of total lifetime resources from the *****
% ********* data. This is why we set these values up front.) *************
% ************************************************************************


%preference parameters
rho = .025;
eta = 0.23;
chi = 12*2.5*230;  % Benchmark calibration for subsistence consumption
chivec = 12*2.5*[230; 300; 150];  % other levels of subsistence concumption
                                  % for performing the sensitivity analysis  

%returns
rf = .03;
R = .07;

%shocks
sigma = 0.2;



% ************************************************************************
% ************************************************************************
% *********Replication of Table A2 fron the Online Data Appendix**********
% ************************************************************************


TableA2 = zeros(6,4);  % Initialize a 6x4 matrix that will be filled with 
                       % entries of Table A2


load Mean_Value_of_Financial_Assets.dat;
load Family_Net_Worth.dat;


% Generate Column (1) of Table A2 (see this table's notes in the Online 
% Data Appendix).
TableA2(:,1) = Mean_Value_of_Financial_Assets'./Family_Net_Worth';


% Generate Column (2) of Table A2 (see this table's notes in the Online 
% Data Appendix).
load Stock_Holdings_as_Share_of_Financial_Assets.dat;

TableA2(:,2) = TableA2(:,1).*Stock_Holdings_as_Share_of_Financial_Assets'; 


% Generate Column (3) of Table A2 (see this table's notes in the Online 
% Data Appendix).
TableA2(:,3) = 0.2872*Stock_Holdings_as_Share_of_Financial_Assets'/(sum(Stock_Holdings_as_Share_of_Financial_Assets)/6);

% Generate Column (4) of Table A2 (see this table's notes in the Online 
% Data Appendix).
TableA2(:,4) = TableA2(:,2) + TableA2(:,3);

Fraction_of_net_worth_held_in_stocks_or_equity = TableA2(:,4); % This is
                    % column (5) of Table 2 in the paper. Store it here at
                    %high precision before numbers are rounded (see below).


TableA2 = 100*TableA2;  %All entries of Table A2 appear as percentages

% Round all Table entries to the nearest second decimal
TableA2 = round(100*TableA2)/100;

format short g;
% Display Table A2
disp('Table A2 from the Online Data Appendix');
disp(TableA2);


% ************************************************************************
% **** END Replication of Table A2 fron the Online Data Appendix**********
% ************************************************************************
% ************************************************************************




% ************************************************************************
% ************************************************************************
% *****************Replication of Table 2 in the paper********************
% ************************************************************************

Table2 = zeros(6,10);  % Initialize a 6x10 matrix that will be filled with 
                        % entries of Table 2

% Generate Column (1) of Table 2 (see this table's notes)                        
Table2(:,1) = Family_Net_Worth';

% Generate Column (2) of Table 2 (see this table's notes)                        
load Before_Tax_Family_Income.dat;
Table2(:,2) = Before_Tax_Family_Income';

% Generate Column (3) of Table 2 (see this table's notes)                        
load Effective_Marginal_Tax_Rate.dat;
Table2(:,3) = Effective_Marginal_Tax_Rate';


% Generate Column (4) of Table 2 (see this table's notes)
Table2(:,4)  = Table2(:,2).*(1-Table2(:,3));

% Generate Column (5) of Table 2 (see this table's notes and also details 
% in this program above, regarding the construction of Table A2 in the 
% Online Data Appendix)
Table2(:,5)  = Fraction_of_net_worth_held_in_stocks_or_equity;

% Generate Column (6) of Table 2 (read this table's notes carefully)
% Notice that the model's calibrated returns, rf and R, which are 
% declared above affect this calculation

Table2(:,6)  = Table2(:,1) + (Table2(:,4)./(Table2(:,5)*R+(1-Table2(:,5))*rf)).*(1-1./(1+(Table2(:,5)*R+(1-Table2(:,5))*rf)).^78);

k = 1000*Table2(:,6);   % These are key entries of total resources needed 
                        % for all simulation exercises (multiplied by 1000)
                        % because Column (6) of Table 2 appears in 
                        % thousands of 2007 USD)


% Generate Column (7) of Table 2 (read this table's notes carefully and the
% text in the paper)
Table2(:,7)  = Table2(:,1).*Table2(:,5)./Table2(:,6);

Phik = Table2(:,7);   % These are key data to be matched (see Figure 1)


% Generate Column (8) of Table 2 (read the equations of Proposition 1 in 
% the paper)
Table2(:,8) = eta*(R-rf)/sigma^2*(1-(chi/rf)./k);


% Generate Column (9) of Table 2 (uses the equation appearing right after 
% equation (8) in the paper)
zeta = eta*((eta+1)/2*((R-rf)/sigma)^2+rf-rho);
phi0=eta*((R-rf)/sigma)^2;

Table2(:,9) = zeta./(phi0 + rf./(1-(chi/rf)./k));


% Generate Column (10) of Table 2 (uses the formula for the coefficient of 
% the EIS)

ksi = (1/2)*((R-rf)/sigma)^2;
ksic = rho*eta +(1-eta)*rf - eta*(eta-1)*ksi;

cvec = ksic*(k-chi/rf) + chi;
Table2(:,10) = eta*(1-chi./cvec);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Display Table 2
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

Table2(:,3) = 100*Table2(:,3);  % Entries of column (3) appear as 
                                % percentages
                                
Table2(:,5) = 100*Table2(:,5);  % Entries of column (5) appear as 
                                % percentages

Table2(:,7) = 100*Table2(:,7);  % Entries of column (7) appear as 
                                % percentages

Table2(:,8) = 100*Table2(:,8);  % Entries of column (8) appear as 
                                % percentages

Table2(:,9) = 100*Table2(:,9);  % Entries of column (9) appear as 
                                % percentages
                                
% Round all Table entries to the nearest second decimal
Table2 = round(100*Table2)/100;

format short g;

disp('Table 2 from the Paper');
disp(Table2);



% ************************************************************************
% ************ END Replication of Table 2 in the paper********************
% ************************************************************************
% ************************************************************************



% ************************************************************************
% ************************************************************************
% *********** Replication of Figures 1 and 2 in the paper ****************
% ************************************************************************

zeta = eta*((eta+1)/2*((R-rf)/sigma)^2+rf-rho);
phi0=eta*((R-rf)/sigma)^2;


Figure1 = zeros(6,4);  % Initialize a 6x4 matrix that will be filled with 
                       % entries of Figure 1


                       
Figure2 = zeros(6,3);  % Initialize a 6x3 matrix that will be filled with 
                       % entries of Figure 2

                       
                       
for j = 1:3;

Figure1(:,j) = 100*eta*(R-rf)/sigma^2*(1-(chivec(j)/rf)./k);

Figure2(:,j) = 100*zeta./(phi0 + rf./(1-(chivec(j)/rf)./k));

end;

Figure1(:,4) = 100*Phik;

y = [10 30 50 70 85 100];

subplot(221), plot(y,Figure1), title('Figure 1');
xlabel('Income Categories'), ylabel('Fraction of Risky Assets Held');

subplot(223), plot(y,Figure2), title('Figure 2');
xlabel('Income Categories'), ylabel('Saving Rates');



% ************************************************************************
% ********* END Replication of Figures 1 and 2 in the paper **************
% ************************************************************************
% ************************************************************************



% ************************************************************************
% ************************************************************************
% **************** Replication of Table 3 in the paper********************
% ************************************************************************

Table3 = zeros(6,3);  % Initialize a 6x3 matrix that will be filled with 
                       % entries of Table 3

ksi = (1/2)*((R-rf)/sigma)^2; 
kvec = [322.58; 713.41; 1087.98; 1548.89; 2119.78; 6305.27];
ksic = rho*eta +(1-eta)*rf - eta*(eta-1)*ksi;

for j = 1:3;

cvec = ksic*(1000*kvec-chivec(j)/rf) + chivec(j);
EISvec = eta*(1-chivec(j)./cvec);

Table3(:,j)=1./EISvec;

% disp('EISvec');
% disp(EISvec);
% disp('Coeff Rel. Risk Aversion');
% disp(crel);

end;

% Round all Table entries to the nearest second decimal
Table3 = round(100*Table3)/100;

format short g;

disp('Table 3 from the Paper');
disp(Table3);


% ************************************************************************
% ************ END Replication of Table 3 in the paper********************
% ************************************************************************
% ************************************************************************

