import os  # allow imports
import sys

parent_dir = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
for module_folder in ["algorithms", "benchmark", "zx_dfs", "statistics", "bin"]:
    sys.path.append(f"{parent_dir}/{module_folder}")
else:
    sys.path.append(f"{parent_dir}")

from qiskit.transpiler.basepasses import TransformationPass
from qiskit.converters import circuit_to_dag, dag_to_circuit
from qiskit import QuantumCircuit
from qiskit.dagcircuit import DAGCircuit

from zx_dfs.dfs import DFS
from zx_dfs.idfs import IDFS

# from zx_dfs.bfs import BFS
from zx_dfs.dfs import AllResults, Result


class QiskitCompilerPass(TransformationPass):
    def __init__(self, optimization_function: DFS | IDFS) -> None:
        super().__init__()
        self.optimization_function = optimization_function

    def run(self, dag: DAGCircuit) -> DAGCircuit:

        # convert dag to quantum circuit
        qc: QuantumCircuit = dag_to_circuit(dag)

        #  get arguments that the optimization Class was initialized
        opti_args: dict = self.optimization_function.my_init_arguments
        opti_args["qc"] = qc  # add quantum circuit back to the arguments

        # reconstruct object and call __init__ with updated arguments
        self.optimization_function.__init__(**opti_args)

        # run optimization
        all_results: AllResults = self.optimization_function.run()

        # get the results with the best circuit for the first metric
        metric = opti_args["metric"][0]
        metric_result: Result = getattr(all_results, str(metric))

        # convert quantum circuit to dag
        qc_opt = metric_result.qc
        dag_opt = circuit_to_dag(qc_opt)

        return dag_opt
