"""
Iterative Deepening DFS
"""

import sys
import qiskit as qk
from qiskit import QuantumCircuit
from qiskit.transpiler import PassManager
from qiskit.transpiler.passes import Unroll3qOrMore

import pyzx as zx
from pyzx.graph.base import BaseGraph, VT, ET
from pyzx.circuit import Circuit
from pyzx import extract_circuit
from pyzx.simplify import to_graph_like, is_graph_like

from typing import Tuple, Callable, Any, Self

import queue
from queue import LifoQueue, Queue

from copy import deepcopy
from copy import copy

from multiprocessing import Process, Lock, cpu_count, Value, shared_memory, Manager
from multiprocessing.managers import BaseManager

from .full_analysis import full_analysis

from algorithms.pyzx import pyzx_full_reduce

from dataclasses import dataclass, make_dataclass, fields
import dill
import datetime as DT
from datetime import datetime, timedelta

from .dfs import DFS, Result, AllResults, get_circuit_statistics, zx_rules


class IDFS(DFS):
    def run(self) -> AllResults:
        max_level: int = 1

        b_iterative_deepening: bool = True

        # compute maximum allowed runtime of dfs
        if self.b_max_duration:
            self.end_time: datetime = datetime.now() + timedelta(
                seconds=self.max_duration
            )

        TMPResults = self._setup_results()

        while b_iterative_deepening:
            dic_root: dict = self._setup()
            if not is_graph_like(dic_root["g"]):
                to_graph_like(dic_root["g"])

            TMPResults = deepcopy(self.AllResults)
            TMPTree = deepcopy(self.tree)

            self.tree = self._setup_tree()
            self.AllResults = self._setup_results()

            s: LifoQueue = LifoQueue()  # LIFO is stack for dfs
            s.put(deepcopy(dic_root))

            b_iterative_deepening = False

            # compute initial time for first snapshot
            if self.b_snapshot:
                prefix_cntr: int = 0

                # dump initial snapshot and update all results
                if self.b_snapshot_pkl:
                    self._snapshot_results(prefix_cntr)

                # generate initial snapshot csv file with solution overview
                time_stamp: datetime = datetime.now()
                self._snapshot_csv(prefix_cntr, time_stamp, force=True)

                # generate time when next snapshot should be executed
                next_snapshot: datetime = datetime.now() + timedelta(
                    seconds=self.snapshot_frequency
                )

            while s.qsize() > 0:

                dic_stack = s.get()

                if not is_graph_like(dic_stack["g"]):
                    to_graph_like(dic_stack["g"])

                # terminate dfs if maximum allowed runtime was reached
                if self.b_max_duration:
                    self.current_time: datetime = datetime.now()
                    if self.current_time > self.end_time:
                        self.AllResults = TMPResults
                        self.tree = TMPTree
                        break

                # execute snapshot
                if self.b_snapshot:
                    time_stamp: datetime = datetime.now()
                    if time_stamp > next_snapshot:
                        # dump snapshot and update all results
                        if self.b_snapshot_pkl:
                            self._snapshot_results(prefix_cntr)

                        # generate snapshot csv file with solution overview
                        self._snapshot_csv(prefix_cntr, time_stamp)

                        # update when next snapshot should be executed
                        next_snapshot: datetime = datetime.now() + timedelta(
                            seconds=self.snapshot_frequency
                        )

                        # update prefix_cntr
                        prefix_cntr += 1

                for rule in self.l_zx_rules:

                    # get a dictionary with all parameters of currentliy visited node
                    dic_child, node = self._get_current_node(rule, dic_stack)

                    # get depth of current of current node
                    if self.b_trace_depth:
                        current_depth = dic_child["cntr_max_depth"]
                    else:
                        current_depth = None

                    # apply rule to graph
                    child_graph = deepcopy(dic_stack["g"])
                    cntr = rule(child_graph, quiet=self.b_quiet)

                    # explore node propterties at every node of the tree
                    if self.b_check_every_node:
                        self._check_metric(
                            child_graph, node, current_depth, self.metric
                        )

                    # only deepen to next level if we changed the graph at the current max level
                    if (dic_child["cntr_max_depth"] == max_level) and (cntr > 0):
                        b_iterative_deepening = True
                        if self.b_trace_leafs:
                            self.n_leafs += 1

                            self._check_metric(
                                child_graph, node, current_depth, self.metric
                            )

                    # push to stack if rule can still be applied
                    elif cntr > 0:
                        dic_child["g"] = child_graph
                        if not self._check_pruning_conditions(dic_child, dic_stack):
                            s.put(dic_child)

                    # explore node properties at leaf node
                    else:
                        if self.b_trace_leafs:
                            self.n_leafs += 1

                            self._check_metric(
                                child_graph, node, current_depth, self.metric
                            )

            # terminate dfs if maximum allowed runtime was reached
            if self.b_max_duration:
                self.current_time: datetime = datetime.now()
                if self.current_time > self.end_time:
                    for i, m in enumerate(self.metric):
                        best_result = getattr(TMPResults, m.__str__())
                        self._check_metric(
                            best_result.graph,
                            best_result.node,
                            best_result.depth,
                            self.metric,
                        )
                    self.tree = TMPTree
                    b_iterative_deepening = False
                    break

            # only deepen if tree is not finished
            if b_iterative_deepening:
                max_level += 1

        # update if solution globally optimal or not
        self._finalize_results()

        # compute initial time for first snapshot
        if self.b_snapshot:
            # differentiate between inital and final snapshot if dfs finishes inside the first window
            if prefix_cntr == 0:
                prefix_cntr = 1

            # dump initial snapshot and update all results
            if self.b_snapshot_pkl:
                self._snapshot_results(prefix_cntr)

            # generate initial snapshot csv file with solution overview
            time_stamp: datetime = datetime.now()
            self._snapshot_csv(prefix_cntr, time_stamp)

        # update and save results from dfs
        self._save_results()

        return self.AllResults
