from typing import List, Callable, Optional, Union, Generic, Tuple, Dict, Iterator, cast

from pyzx.utils import EdgeType, VertexType, toggle_edge, vertex_is_zx, toggle_vertex
from pyzx.graph.base import BaseGraph, VT, ET
from pyzx.circuit import Circuit
from pyzx.simplify import Stats
from pyzx.rules import MatchObject, RewriteOutputType

from pyzx.simplify import (
    pivot_simp,
    pivot_gadget_simp,
    pivot_boundary_simp,
    lcomp_simp,
    bialg_simp,
    spider_simp,
    id_simp,
    gadget_simp,
    supplementarity_simp,
    copy_simp,
    phase_free_simp,
)

from .colour_change import to_gh


def full_analysis(
    g: BaseGraph[VT, ET],
    matchf: Optional[Callable[[Union[VT, ET]], bool]] = None,
    quiet: bool = True,
    stats: Optional[Stats] = None,
) -> int:
    """The main simplification routine of PyZX. It uses a combination of :func:`clifford_simp` and
    the gadgetization strategies :func:`pivot_gadget_simp` and :func:`gadget_simp`."""

    # counter of how often colour changing rules were used
    n_cntr_colour: int = 0

    if any(g.types()[h] == VertexType.H_BOX for h in g.vertices()):
        raise ValueError(
            "Input graph is not a ZX-diagram as it contains an H-box. "
            "Maybe call pyzx.hsimplify.from_hypergraph_form(g) first?"
        )
    k, n_cntr_colour = interior_clifford_simp(
        g, matchf=matchf, quiet=quiet, stats=stats, n_cntr_colour=n_cntr_colour
    )
    pivot_gadget_simp(g, matchf=matchf, quiet=quiet, stats=stats)
    while True:
        k, n_cntr_colour = clifford_simp(
            g, matchf=matchf, quiet=quiet, stats=stats, n_cntr_colour=n_cntr_colour
        )
        i = gadget_simp(g, quiet=quiet, stats=stats)
        k, n_cntr_colour = interior_clifford_simp(
            g, matchf=matchf, quiet=quiet, stats=stats, n_cntr_colour=n_cntr_colour
        )
        j = pivot_gadget_simp(g, matchf=matchf, quiet=quiet, stats=stats)
        if i + j == 0:
            break
    return n_cntr_colour


def interior_clifford_simp(
    g: BaseGraph[VT, ET],
    n_cntr_colour: int,
    matchf: Optional[Callable[[Union[VT, ET]], bool]] = None,
    quiet: bool = False,
    stats: Optional[Stats] = None,
) -> (int, int):
    """Keeps doing the simplifications ``id_simp``, ``spider_simp``,
    ``pivot_simp`` and ``lcomp_simp`` until none of them can be applied anymore."""
    spider_simp(g, matchf=matchf, quiet=quiet, stats=stats)

    changed: int = to_gh(g)
    n_cntr_colour = update_colour_cntr(n_cntr_colour, changed)

    i = 0
    while True:
        i1 = id_simp(g, matchf=matchf, quiet=quiet, stats=stats)
        i2 = spider_simp(g, matchf=matchf, quiet=quiet, stats=stats)
        i3 = pivot_simp(g, matchf=matchf, quiet=quiet, stats=stats)
        i4 = lcomp_simp(g, matchf=matchf, quiet=quiet, stats=stats)
        if i1 + i2 + i3 + i4 == 0:
            break
        i += 1
    return i, n_cntr_colour


def update_colour_cntr(n_cntr_colour: int, n_cntr_current: int) -> int:
    if n_cntr_current > 0:
        n_cntr_colour += 1

    return n_cntr_colour


def clifford_simp(
    g: BaseGraph[VT, ET],
    n_cntr_colour: int,
    matchf: Optional[Callable[[Union[VT, ET]], bool]] = None,
    quiet: bool = True,
    stats: Optional[Stats] = None,
) -> (int, int):
    """Keeps doing rounds of :func:`interior_clifford_simp` and
    :func:`pivot_boundary_simp` until they can't be applied anymore."""
    i: int = 0
    while True:
        k: int = 0
        k, n_cntr_colour = interior_clifford_simp(
            g, matchf=matchf, quiet=quiet, stats=stats, n_cntr_colour=n_cntr_colour
        )
        i += k
        i2 = pivot_boundary_simp(g, matchf=matchf, quiet=quiet, stats=stats)
        if i2 == 0:
            break
    return i, n_cntr_colour


def simp(
    g: BaseGraph[VT, ET],
    name: str,
    match: Callable[..., List[MatchObject]],
    rewrite: Callable[
        [BaseGraph[VT, ET], List[MatchObject]], RewriteOutputType[ET, VT]
    ],
    matchf: Optional[Union[Callable[[ET], bool], Callable[[VT], bool]]] = None,
    quiet: bool = True,
    stats: Optional[Stats] = None,
) -> int:
    """Helper method for constructing simplification strategies based on the rules present in rules_.
    It uses the ``match`` function to find matches, and then rewrites ``g`` using ``rewrite``.
    If ``matchf`` is supplied, only the vertices or edges for which matchf() returns True are considered for matches.

    Example:
        ``simp(g, 'spider_simp', rules.match_spider_parallel, rules.spider)``

    Args:
        g: The graph that needs to be simplified.
        str name: The name to display if ``quiet`` is set to False.
        match: One of the ``match_*`` functions of rules_.
        rewrite: One of the rewrite functions of rules_.
        matchf: An optional filtering function on candidate vertices or edges, which
           is passed as the second argument to the match function.
        quiet: Suppress output on numbers of matches found during simplification.

    Returns:
        Number of iterations of ``rewrite`` that had to be applied before no more matches were found.
    """

    i = 0
    new_matches = True
    while new_matches:
        new_matches = False
        if matchf is not None:
            m = match(g, matchf)
        else:
            m = match(g)
        if len(m) > 0:
            i += 1
            # if i == 1 and not quiet: print("{}: ".format(name),end='')
            # if not quiet: print(len(m), end='')
            # print(len(m), end='', flush=True) #flush only supported on Python >3.3
            etab, rem_verts, rem_edges, check_isolated_vertices = rewrite(g, m)
            g.add_edge_table(etab)
            g.remove_edges(rem_edges)
            g.remove_vertices(rem_verts)
            if check_isolated_vertices:
                g.remove_isolated_vertices()
            # if not quiet: print('. ', end='')
            # print('. ', end='', flush=True)
            new_matches = True
            if stats is not None:
                stats.count_rewrites(name, len(m))
    # if not quiet and i>-1: print(' {!s} iterations'.format(i))
    print(f"{name},{rewrite.__name__},{i}")
    return i
