"""
DFS
"""

import qiskit as qk
from qiskit import QuantumCircuit
from qiskit.transpiler import PassManager
from qiskit.transpiler.passes import Unroll3qOrMore

import pyzx as zx
from pyzx.graph.base import BaseGraph, VT, ET
from pyzx.circuit import Circuit
from pyzx import extract_circuit, tcount
from pyzx.simplify import (
    to_graph_like,
    is_graph_like,
    interior_clifford_simp,
    clifford_simp,
)

from typing import Tuple, Callable, Any, Self, TypeVar

import queue
from queue import LifoQueue, Queue

from copy import deepcopy
from copy import copy

from multiprocessing import Process, Lock, cpu_count, Value, shared_memory, Manager
from multiprocessing.managers import BaseManager

from .full_analysis import full_analysis

from algorithms.pyzx import pyzx_full_reduce

from benchmark.statistics import (
    get_circuit_statistics as get_circuit_statistics_benchmark,
)

import pandas as pd
from pandas import DataFrame

from dataclasses import dataclass, make_dataclass, fields
import dill
import datetime as DT
from datetime import datetime, timedelta

import inspect

from .metric import Metric, MetricTcount

from .pruning import PruneColourCycle, PruneColourChange, PruneMaxDepth


@dataclass
class Result:
    metric: str
    graph: BaseGraph[VT, ET]
    circuit: Circuit | None
    qc: QuantumCircuit | None
    node: int | None
    depth: int
    l_path: list[int] | None
    l_rule_sequence: list[str] | None
    l_bundled_rules: list[Callable] | None
    expired_time: float | None


@dataclass
class AllResults:
    metric: Result | None = None
    n_nodes: int | None = 0
    n_leafs: int | None = 0
    max_depth: int | None = 0
    tree: dict[int, Tuple[None | int, None | str]] | None = None
    optimal: bool = False
    qc_init: QuantumCircuit | None = None
    c_init: Circuit | None = None
    g_init: BaseGraph[VT, ET] | None = None


class DFS:
    def __init__(
        self,
        qc: QuantumCircuit = QuantumCircuit(),
        metric: list = [MetricTcount()],
        pruning: list = [PruneColourCycle()],
        b_quiet: bool = True,
        b_trace_leafs: bool = True,
        b_trace_nodes: bool = True,
        b_trace_depth: bool = True,
        b_check_every_node: bool = True,
        mult: int = 2,
        max_depth: int = 2,
        l_zx_rules: None | list[Callable] = None,
        b_full_start: bool = False,
        circ_name: str | None = None,
        b_dump_results: bool = True,
        func_opt: str | None = None,
        b_max_duration: bool = True,
        max_duration: int = 60 * 60 * 6,  # maximum duration in seconds
        b_snapshot: bool = True,  # for csv files
        b_snapshot_pkl: bool = False,  # for pickles objects with graphs
        snapshot_frequency: int = 1,  # snapshot every n seconds
    ) -> None:
        """
        setup and input circuits
        """
        # init arguments for needed for compiler pass
        self.my_init_arguments = self._get_init_arguments()

        # graphs and inputs
        self.qc_init: QuantumCircuit = self._unroll_qc_2q(qc)
        t_root_node: Tuple[BaseGraph[VT, ET], Circuit] = self._import_circuit(
            self._unroll_qc_2q(qc)
        )

        self.g_init: BaseGraph[VT, ET] = deepcopy(t_root_node[0])
        self.c_init: Circuit = t_root_node[1]

        # the metric to optimize for
        self.metric = metric
        self.pruning = pruning

        # data class that stores our results
        self.AllResults = self._setup_results()

        # circ name is required for filename if we dump our result object
        self.circ_name: str | None = circ_name

        # whether to dump result object
        self.b_dump_results: bool = b_dump_results
        # identifier for pickle object
        self.func_opt: str | None = func_opt

        """
        tracing and dfs behaviour
        """
        # no output if rules change graph
        self.b_quiet: bool = b_quiet

        # trace dfs
        self.b_trace_leafs: bool = b_trace_leafs
        self.b_trace_nodes: bool = b_trace_nodes
        self.b_trace_depth: bool = b_trace_depth

        # properties for tracing
        self.n_leafs: int | None = None
        self.n_nodes: int | None = None

        # dfs depth
        self.depth: int = 0

        # check whether to check every node
        self.b_check_every_node: bool = b_check_every_node

        # snapshot run every n seconds
        self.b_snapshot: bool = b_snapshot
        self.snapshot_frequency: int = snapshot_frequency
        self.b_snapshot_pkl: bool = b_snapshot_pkl

        """
        pruning rules
        """
        self.mult: int = mult
        self.max_depth: int = max_depth

        self.b_full_start: bool = b_full_start

        # limit the time a dfs instance is allowed to run
        self.b_max_duration: bool = b_max_duration
        self.max_duration: int = max_duration  # seconds

        # use default rule set or custom rules
        if l_zx_rules is None:
            self.l_zx_rules = zx_rules()
        else:
            self.l_zx_rules = l_zx_rules

        # set up dictionary for path setup
        self.tree: dict = self._setup_tree()

        self.start_time = DT.datetime.now()

        pass

    def _get_init_arguments(self):

        sig = inspect.signature(self.__init__)
        bound_args = sig.bind_partial(self)
        bound_args.apply_defaults()

        return bound_args.arguments

    # root node does not have parents or rule
    def _setup_tree(self) -> dict:
        # {current_id: (parent_id, current_rule)}
        return {0: (None, None)}

    def _import_circuit(self, qc: QuantumCircuit) -> Tuple[BaseGraph[VT, ET], Circuit]:
        c: Circuit = Circuit.from_qasm(qk.qasm2.dumps(qc))
        g: BaseGraph[VT, ET] = c.to_basic_gates().to_graph()
        to_graph_like(g)
        return g, c

    def _node_skeleton(self) -> dict:
        dic_node = {
            "c": None,
            "g": None,
            "n_colour_change": 0,
            "cntr_colour_change": 0,
            "cntr_max_depth": 0,
            "past_rule": None,
            "max_depth": self.max_depth,
        }
        return dic_node

    def _setup(self) -> dict:

        dic_root = self._node_skeleton()

        if self.b_full_start:
            result = pyzx_full_reduce(self.qc_init, self.circ_name)
            dic_root["g"] = deepcopy(result.c_tcount.graph)
            dic_root["c"] = deepcopy(result.c_tcount.circuit)
        else:
            dic_root["g"] = deepcopy(self.g_init)
            dic_root["c"] = deepcopy(self.c_init)
            dic_root["n_colour_change"] = self._setup_colour_change()
            dic_root["max_depth"] = self.max_depth

        # setup trace
        self._setup_trace()

        # add node id to generate tree if tracing is available
        if self.b_trace_nodes:
            dic_root["node"] = None

        return deepcopy(dic_root)

    def _setup_trace(self):
        if self.b_trace_leafs:
            self.n_leafs = 0
        if self.b_trace_nodes:
            self.n_nodes = 0

    def _setup_colour_change(self) -> int:

        cntr_colour_change: int = full_analysis(deepcopy(self.g_init))

        # apply multiplicator if colour was changed more often, else just add mult as limit
        if cntr_colour_change > 0:
            cntr_colour_change *= self.mult
        else:
            cntr_colour_change = self.mult

        return cntr_colour_change

    def _finalize_results(self) -> AllResults:
        for var in fields(self.AllResults):
            if var.name in [m.__str__() for m in self.metric]:
                old_metric = getattr(self.AllResults, var.name)

                if not is_graph_like(old_metric.graph):
                    to_graph_like(old_metric.graph)
                try:

                    old_metric.circuit = extract_circuit(old_metric.graph.copy())
                    old_metric.qc = QuantumCircuit.from_qasm_str(
                        old_metric.circuit.to_qasm()
                    )
                except:
                    continue

                setattr(self.AllResults, var.name, old_metric)

        AllResults.optimal = False

        return AllResults

    def _setup_results(self) -> AllResults:
        AllResults = make_dataclass(
            "AllResults",
            [(m.__str__(), type(Result)) for m in self.metric]
            + [
                ("n_nodes", int),
                ("n_leafs", int),
                ("max_depth", int),
                ("node", None | int),
                ("tree", None | dict),
                ("qc_init", None | QuantumCircuit),
                ("c_init", None | Circuit),
                ("g_init", None | BaseGraph[VT, ET]),
            ],
        )

        # initialize result data class
        for var in fields(AllResults):
            if (
                var.name == "n_nodes"
                or var.name == "n_leafs"
                or var.name == "max_depth"
            ):
                setattr(AllResults, var.name, 0)
            elif var.name == "tree":
                setattr(AllResults, var.name, None)
            elif var.name == "qc_init":
                setattr(AllResults, var.name, deepcopy(self.qc_init))
            elif var.name == "c_init":
                setattr(AllResults, var.name, deepcopy(self.c_init))
            elif var.name == "g_init":
                setattr(AllResults, var.name, deepcopy(self.g_init))
            else:
                setattr(
                    AllResults,
                    var.name,
                    Result(
                        var.name,
                        deepcopy(self.g_init),
                        None,
                        None,
                        0,
                        0,
                        None,
                        None,
                        None,
                        0.0,
                    ),
                )
        AllResults.optimal = False

        return AllResults

    def _check_pruning_conditions(self, dic_child: dict, dic_parent: dict) -> bool:
        l_skip: list[bool] = [
            p.check(dic_child, dic_parent, self.metric)[0] for p in self.pruning
        ]
        if True in l_skip:
            if self.b_trace_leafs:
                self.n_leafs += 1
            return True
        else:
            return False

    def _snapshot_csv(
        self, prefix_cntr: int, time_stamp: datetime, force: bool = False
    ) -> None:
        l_dic_opt = []

        # force update of result object with current values
        if self.b_trace_nodes:
            self.AllResults.n_nodes = self.n_nodes
            self.AllResults.tree = self.tree
        if self.b_trace_leafs:
            self.AllResults.n_leafs = self.n_leafs
        if self.b_trace_depth:
            self.AllResults.max_depth = self.depth

        for m in self.metric:
            metric_result = getattr(self.AllResults, m.__str__())

            # dic_opt: dict = get_circuit_statistics_benchmark(
            #    qc=metric_result.qc,
            #    c=metric_result.circuit,
            #    g=deepcopy(metric_result.graph),
            #    circuit_name=self.circ_name,
            #    func_opt=self.func_opt,
            #    metric=m.__str__(),
            # )
            dic_opt = {
                "func_opt": self.func_opt,
                "g_edges": metric_result.graph.num_edges(),
                "g_vertices": metric_result.graph.num_vertices(),
                "g_tcount": tcount(metric_result.graph),
                "metric": m.__str__(),
            }

            if self.func_opt is None:
                func_opt: str = f"{self.__class__.__name__}"
            else:
                func_opt: str = self.func_opt

            dic_opt["timestamp"] = time_stamp
            dic_opt["snap_cntr"] = prefix_cntr

            # only add number of nodes and leafs to dictionary if they are present; meaning func_opt is dfs
            if hasattr(self.AllResults, "n_nodes"):
                dic_opt["n_nodes"] = self.AllResults.n_nodes
            if hasattr(self.AllResults, "n_leafs"):
                dic_opt["n_leafs"] = self.AllResults.n_leafs
            if hasattr(self.AllResults, "max_depth"):
                dic_opt["max_depth"] = self.AllResults.max_depth
            if hasattr(metric_result, "node"):
                dic_opt["node"] = metric_result.node
            if hasattr(metric_result, "depth"):
                dic_opt["depth"] = metric_result.depth
            # if hasattr(self.AllResults, "optimal"):
            #    if dic_opt[m.__str__()] == 0:
            #        dic_opt["optimal"] = True
            #    else:
            #        dic_opt["optimal"] = self.AllResults.optimal

            l_dic_opt.append(dic_opt)

        # only overwrite csv for initial snapshot, else apply to file
        df_results: DataFrame = pd.DataFrame(l_dic_opt)
        if prefix_cntr == 0 and force is True:
            df_results.to_csv(
                f"_{self.circ_name}_{func_opt}_checkpoint.csv",
                index=False,
                mode="w",
            )  # write csv
        else:
            df_results.to_csv(
                f"_{self.circ_name}_{func_opt}_checkpoint.csv",
                index=False,
                mode="a",
            )  # write csv

    def _snapshot_results(self, prefix_cntr: int) -> None:
        # save potential trace to results
        if self.b_trace_nodes:
            self.AllResults.n_nodes = self.n_nodes
            self.AllResults.tree = self.tree

            # compute path for best results
            for m in self.metric:
                metric_result = getattr(self.AllResults, m.__str__())

                # get path
                l_path = self._trace_path(metric_result.node)
                metric_result.l_path = l_path

                # get sequence
                l_rule_sequence = [self.tree[i][1].__name__ for i in l_path]
                metric_result.l_rule_sequence = l_rule_sequence

                # update object
                setattr(AllResults, m, metric_result)

        if self.b_trace_leafs:
            self.AllResults.n_leafs = self.n_leafs
        if self.b_trace_depth:
            self.AllResults.max_depth = self.depth

        # save pickled object to  reuse the graphs later on
        if self.b_dump_results:
            self._dump_results(prefix_cntr)

    def _save_results(self) -> None:
        # save potential trace to results
        if self.b_trace_nodes:
            self.AllResults.n_nodes = self.n_nodes
            self.AllResults.tree = self.tree

            # compute path for best results
            for m in self.metric:
                metric_result = getattr(self.AllResults, m.__str__())

                # get path
                l_path = self._trace_path(metric_result.node)
                metric_result.l_path = l_path

                # get sequence
                l_rule_sequence = [
                    self.tree[i][1].__name__
                    for i in l_path
                    if self.tree[i][1] is not None
                ]
                metric_result.l_rule_sequence = l_rule_sequence

                # update object
                setattr(AllResults, m.__str__(), metric_result)

        if self.b_trace_leafs:
            self.AllResults.n_leafs = self.n_leafs
        if self.b_trace_depth:
            self.AllResults.max_depth = self.depth

        # save pickled object to reuse the graphs later on
        if self.b_dump_results:
            self._dump_results()

    # get all information to construct a dict for current node
    def _get_current_node(
        self, rule: Callable, dic_stack: dict
    ) -> Tuple[dict, int | None]:

        dic_child = self._node_skeleton()
        dic_child["past_rule"] = rule.__name__

        # trace nodes and generate memory
        if self.b_trace_nodes:
            self.n_nodes += 1
            node = self.n_nodes
            dic_child["node"] = node

            # generate memory for path tracing
            parent_node = dic_stack["node"]
            self.tree[node] = (parent_node, rule)
        else:
            node = None

        # update depth
        dic_child = self._update_depth(dic_stack, dic_child)

        return dic_child, node

    # compare if node is a solution based on given metrics
    def _check_metric(
        self,
        g_new: BaseGraph[VT, ET],
        node: int | None,
        current_depth: int | None,
        metric: list = [MetricTcount()],
        b_extract: bool = True
    ) -> list[bool]:

        l_better: list[bool] = [False for _ in metric]
        for i, m in enumerate(metric):
            best_result = getattr(self.AllResults, m.__str__())

            g_best = getattr(best_result, "graph")

            # only update results if we are better
            if m.check(g_new, g_best):
                l_better[i] = True  # indicate that we are actually better

                try:
                    if b_extract:
                        zx.extract_circuit(g_new.copy())

                    # create result data class for specific metric
                    setattr(
                        self.AllResults,
                        m.__str__(),
                        Result(
                            m.__str__(),
                            deepcopy(g_new),
                            None,
                            None,
                            node,
                            current_depth,
                            None,
                            None,
                            None,
                            (DT.datetime.now() - self.start_time).total_seconds(),
                        ),
                    )
                except:
                    l_better[i] = False  # indicate that we are actually better


        return l_better

    def _update_colour_count(self, dic_child: dict, dic_parent: dict) -> dict:
        dic_child["n_colour_change"] = dic_parent["n_colour_change"]
        dic_child["cntr_colour_change"] = dic_parent["cntr_colour_change"]

        return dic_child

    def _update_depth(self, dic_parent: dict, dic_child: dict) -> dict:
        dic_child["cntr_max_depth"] = dic_parent["cntr_max_depth"] + 1

        if self.depth < dic_child["cntr_max_depth"]:
            self.depth = dic_child["cntr_max_depth"]

        return dic_child

    # main method of the class that performs the optimization
    def run(self) -> AllResults:

        dic_root: dict = self._setup()

        s: LifoQueue = LifoQueue()  # LIFO is stack for dfs
        s.put(dic_root)

        # compute maximum allowed runtime of dfs
        if self.b_max_duration:
            end_time: datetime = datetime.now() + timedelta(seconds=self.max_duration)

        ## compute initial time for first snapshot
        if self.b_snapshot:
            prefix_cntr: int = 0

            # dump initial snapshot and update all results
            if self.b_snapshot_pkl:
                self._snapshot_results(prefix_cntr)

            # generate initial snapshot csv file with solution overview
            time_stamp: datetime = datetime.now()
            self._snapshot_csv(prefix_cntr, time_stamp, force=True)

            # generate time when next snapshot should be executed
            next_snapshot: datetime = datetime.now() + timedelta(
                seconds=self.snapshot_frequency
            )

        while s.qsize() > 0:
            dic_stack = s.get()

            if not is_graph_like(dic_stack["g"]):
                to_graph_like(dic_stack["g"])

            # terminate dfs if maximum allowed runtime was reached
            if self.b_max_duration:
                current_time: datetime = datetime.now()
                if current_time > end_time:
                    break

            # execute snapshot
            if self.b_snapshot:
                time_stamp: datetime = datetime.now()
                if time_stamp > next_snapshot:
                    # dump snapshot and update all results
                    if self.b_snapshot_pkl:
                        self._snapshot_results(prefix_cntr)

                    # generate snapshot csv file with solution overview
                    self._snapshot_csv(prefix_cntr, time_stamp)

                    # update when next snapshot should be executed
                    next_snapshot: datetime = datetime.now() + timedelta(
                        seconds=self.snapshot_frequency
                    )

                    # update prefix_cntr
                    prefix_cntr += 1

            for rule in self.l_zx_rules:

                # get a dictionary with all parameters of currentliy visited node
                dic_child, node = self._get_current_node(rule, dic_stack)

                # get depth of current of current node
                if self.b_trace_depth:
                    current_depth = dic_child["cntr_max_depth"]
                else:
                    current_depth = None

                # apply rule to graph
                child_graph = deepcopy(dic_stack["g"])
                cntr = rule(child_graph, quiet=self.b_quiet)

                # explore node propterties at every node of the tree
                if self.b_check_every_node:
                    self._check_metric(child_graph, node, current_depth, self.metric)

                # push to stack if rule can still be applied
                if cntr > 0:
                    dic_child["g"] = child_graph
                    if not self._check_pruning_conditions(dic_child, dic_stack):
                        s.put(dic_child)
                    else:
                        if self.b_trace_leafs:
                            self.n_leafs += 1

                        self._check_metric(
                            child_graph, node, current_depth, self.metric
                        )

                # explore node properties at leaf node
                else:
                    if self.b_trace_leafs:
                        self.n_leafs += 1

                    self._check_metric(child_graph, node, current_depth, self.metric)

        ## compute initial time for first snapshot
        if self.b_snapshot:
            # differentiate between inital and final snapshot if dfs finishes inside the first window
            if prefix_cntr == 0:
                prefix_cntr = 1

            # dump initial snapshot and update all results
            if self.b_snapshot_pkl:
                self._snapshot_results(prefix_cntr)

            # generate initial snapshot csv file with solution overview
            time_stamp: datetime = datetime.now()
            self._snapshot_csv(prefix_cntr, time_stamp, force=True)

        # update and save results from dfs
        self._finalize_results()
        self._save_results()

        return self.AllResults

    # verify correctness of optimized circuits with initial circuit
    def verify_circuit(self) -> list[bool]:
        l_equal = []
        for metric in self.metric:
            metric_result: Result = getattr(self.AllResults, metric.__str__())

            # compare linear maps; attention memory intense
            b_equal: bool = zx.compare_tensors(self.c_init, metric_result.circuit)
            l_equal.append(b_equal)

        return l_equal

    # check node
    def _explore_node(
        self,
        child_graph: BaseGraph[VT, ET],
        node: int | None,
        current_depth: int | None,
    ) -> None:

        # if not is_graph_like(child_graph):
        #    to_graph_like(child_graph)

        # try:

        l_better: list[bool] = self._check_metric(
            child_graph,
            node,
            current_depth,
            metric=self.metric,
        )

        # c_opt = extract_circuit(child_graph.copy())

        # qc_opt: QuantumCircuit = QuantumCircuit.from_qasm_str(c_opt.to_qasm())

        ## expand to single and two qubit gates
        # qc_opt: QuantumCircuit = self._unroll_qc_2q(qc_opt)

        # for better,m in zip(l_better, self.metric):

        #    # create result data class for specific metric
        #    setattr(
        #        self.AllResults,
        #        m.__str__(),
        #        Result(
        #            m.__str__(),
        #            deepcopy(g_new),
        #            None,
        #            None,
        #            node,
        #            current_depth,
        #            None,
        #            None,
        #            None,
        #            (DT.datetime.now() - self.start_time).total_seconds(),
        #        ),
        #    )

        # except Exception:
        #    pass

    # ensure that we get comparable results and expand every quantum circuit into single and two qubit gates
    def _unroll_qc_2q(self, qc: QuantumCircuit) -> QuantumCircuit:
        pm: PassManager = PassManager([Unroll3qOrMore()])
        qc_unrolled: QuantumCircuit = pm.run(qc)
        return qc_unrolled

    # recursively traverse tree dict from start node to root node
    def _trace_path(self, start_id: int) -> list[int]:

        # start node in list
        l_path: list[int] = [start_id]

        # the current node is the dict key while the parent is the dict value inside the tuple
        # if the parent value is None, root node is reached
        while l_path[-1] is not None:
            # add parent to list
            l_path.append(self.tree[l_path[-1]][0])

        l_path.pop()  # remove the last value of this list; its always None and indicates the root node
        l_path.reverse()  # reverse the path for the correct ordering

        return l_path

    def _dump_results(self, prefix: None | int = None) -> None:
        # save our results as a pickled object
        if prefix is not None:
            time_prefix: str = str(prefix)
        else:
            time_prefix: str = DT.datetime.now().strftime("%Y-%m-%d-%H-%M-%S")

        if self.func_opt is None:
            func_opt: str = f"{self.__class__.__name__}"
        else:
            func_opt = self.func_opt

        if isinstance(self.circ_name, str):
            dill.dump(
                self.AllResults,
                open(f"_{time_prefix}_{self.circ_name}_{func_opt}.pkl", "wb"),
                recurse=True,
            )
        else:
            dill.dump(
                self.AllResults,
                open(f"_{time_prefix}_{self.circ_name}_{func_opt}.pkl", "wb"),
                recurse=True,
            )


def zx_rules() -> list[Callable]:
    # import rewrting rules from pyzx
    from pyzx.simplify import (
        pivot_simp,
        pivot_gadget_simp,
        pivot_boundary_simp,
        lcomp_simp,
        bialg_simp,
        spider_simp,
        id_simp,
        gadget_simp,
        supplementarity_simp,
        copy_simp,
        phase_free_simp,
    )

    # use custom rule that return whether a graph was changed or not
    from .colour_change import (
        to_gh,
        to_rg,
    )

    from rewrite_rules import w_fusion_simp, z_to_z_box_simp

    # all rules in a list
    l_rules: list[Callable] = [
        #interior_clifford_simp,
        #clifford_simp,
        pivot_simp,
        pivot_gadget_simp,
        pivot_boundary_simp,
        lcomp_simp,
        spider_simp,
        bialg_simp,
        id_simp,
        gadget_simp,
        supplementarity_simp,
        copy_simp,
        phase_free_simp,
        # w_fusion_simp,
        # z_to_z_box_simp,
        to_gh,
        # to_rg,
    ]

    return l_rules


def get_circuit_statistics(
    qc: QuantumCircuit, c: Circuit, g: BaseGraph[VT, ET]
) -> dict:

    # qiskit circuit properties
    num_qubits: int = qc.num_qubits
    counted_ops: dict = qc.count_ops()
    all_ops: int = qc.size()
    depth: int = qc.depth()

    # circuit properties pyzx
    c_basic: Circuit = c.to_basic_gates()
    tcount: int = c_basic.tcount()
    gates: int = len(c_basic.gates)

    # graph properties
    edges: int = g.num_edges()
    vertices: int = g.num_vertices()

    dic_properties: dict = {
        "q_num_qubits": num_qubits,
        "q_depth": depth,
        "q_all_ops": all_ops,
        "c_tcount": tcount,
        "c_gates": gates,
        "g_edges": edges,
        "g_vertices": vertices,
    }

    # flatten ops to gate count
    for k, v in zip(counted_ops.keys(), counted_ops.values()):
        dic_properties[f"q_{k}"] = v

    return dic_properties
