from typing import List, Callable, Optional, Union, Generic, Tuple, Dict, Iterator, cast

from pyzx.utils import EdgeType, VertexType, toggle_edge, vertex_is_zx, toggle_vertex
from pyzx.graph.base import BaseGraph, VT, ET
from pyzx.circuit import Circuit


def to_gh(g: BaseGraph[VT, ET], quiet: bool = True) -> int:
    """Turns every red node into a green node by changing regular edges into hadamard edges"""
    graph_changed: int = 0

    ty = g.types()
    for v in g.vertices():
        if ty[v] == VertexType.X:
            graph_changed += 1
            g.set_type(v, VertexType.Z)
            for e in g.incident_edges(v):
                et = g.edge_type(e)
                g.set_edge_type(e, toggle_edge(et))

    return graph_changed


def to_rg(
    g: BaseGraph[VT, ET],
    select: Optional[Callable[[VT], bool]] = None,
    quiet: bool = True,
) -> int:
    """Turn green nodes into red nodes by color-changing vertices which satisfy the predicate ``select``.
    By default, the predicate is set to greedily reducing the number of Hadamard-edges.
    :param g: A ZX-graph.
    :param select: A function taking in vertices and returning ``True`` or ``False``."""
    graph_changed: int = 0

    if select is None:
        select = lambda v: (
            len([e for e in g.incident_edges(v) if g.edge_type(e) == EdgeType.SIMPLE])
            < len(
                [e for e in g.incident_edges(v) if g.edge_type(e) == EdgeType.HADAMARD]
            )
        )

    ty = g.types()
    for v in g.vertices():
        if select(v) and vertex_is_zx(ty[v]):
            graph_changed += 1
            g.set_type(v, toggle_vertex(ty[v]))
            for e in g.incident_edges(v):
                g.set_edge_type(e, toggle_edge(g.edge_type(e)))
    return graph_changed
