"""
STATISTICS
"""

import qiskit as qk
from qiskit import QuantumCircuit
from typing import Callable

from pyzx.graph.base import BaseGraph, VT, ET
from pyzx.circuit import Circuit
import pyzx as zx

from zx_dfs.metric import MetricTwoQubit, MetricTcount

"""
get name of circuit
"""


def get_circuit_name(f_qasm: str) -> str:
    circ_name: str = f_qasm.split("/")[-1].split(".qasm")[0]
    return circ_name


"""
prepend "g_" on all gate identifiers, for data processing
"""


def rename_gates_statistics(dic_gates: dict) -> dict:
    dic_post_processing: dict = {}

    for k, v in zip(dic_gates.keys(), dic_gates.values()):
        dic_post_processing["g_" + k] = v

    return dic_post_processing


"""
get the most important circuit properties of a qiskit circuit
"""


def get_circuit_statistics(
    qc: QuantumCircuit,
    c: Circuit,
    g: BaseGraph[VT, ET],
    circuit_name: str | None = None,
    func_opt: Callable | str | None = None,
    metric: str | None = None,
) -> dict:

    # graph properties
    g_tcount: int = zx.tcount(g)
    edges: int = g.num_edges()
    vertices: int = g.num_vertices()

    # qiskit circuit properties
    try:
        num_qubits: int = qc.num_qubits
        counted_ops: dict = qc.count_ops()
        all_ops: int = qc.size()
        depth: int = qc.depth()
    except:
        num_qubits = None
        counted_ops = None
        all_ops = None
        depth = None

    # circuit properties pyzx
    try:
        c_basic: Circuit = c.to_basic_gates()
        tcount: int = c_basic.tcount()
        gates: int = len(c_basic.gates)
        c_two_q: int = MetricTwoQubit()._metric_count(g)
    except:
        c_basic = None
        tcount = None
        gates = None
        c_two_q = None

    if isinstance(func_opt, Callable):
        func_opt = func_opt.__name__

    dic_properties: dict = {
        "circuit_name": circuit_name,
        "func_opt": func_opt,
        "q_num_qubits": num_qubits,
        "q_depth": depth,
        "q_all_ops": all_ops,
        "c_tcount": tcount,
        "c_gates": gates,
        "c_two_q": c_two_q,
        "g_tcount": g_tcount,
        "g_edges": edges,
        "g_vertices": vertices,
        "metric": metric,
    }

    # flatten ops to gate count
    try:
        for k, v in zip(counted_ops.keys(), counted_ops.values()):
            dic_properties[f"q_{k}"] = v
    except:
        pass

    return dic_properties
