import os  # allow imports
import sys
from typing import Tuple

from pyzx.graph.base import BaseGraph, VT, ET
from pyzx.circuit import Circuit

parent_dir = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
for module_folder in ["algorithms", "benchmark", "zx_dfs", "statistics", "bin"]:
    sys.path.append(f"{parent_dir}/{module_folder}")
else:
    sys.path.append(f"{parent_dir}")

from qiskit import QuantumCircuit

from benchmark.run_benchmark import get_circuit_statistics, get_circuit_name

from zx_dfs.metric import MetricEdge, MetricTcount, MetricTcountCircuit, MetricTwoQubit
from zx_dfs.pruning import (
    PruneOptimal,
    PruneMaxDepth,
    PruneColourCycle,
    PruneNonCircuits,
    PruneWorseCircuits,
    PruneColourChange,
    PruneMaxDepth,
)

from zx_dfs.dfs import (
    DFS,
    AllResults,
)


def dfs(
    qc: QuantumCircuit,
    circ_name: str | None,
    metric: list = [MetricTcount()],
    func_opt: str | None = None,
    max_duration=60 * 60 * 6,
) -> AllResults:
    return DFS(
        qc,
        circ_name=circ_name,
        metric=metric,
        pruning=[],
        func_opt=func_opt,
        max_duration=max_duration,
    ).run()


def dfs_no_colour_cycle(
    qc: QuantumCircuit,
    circ_name: str | None,
    metric: list = [MetricTcount()],
    func_opt: str | None = None,
    max_duration=60 * 60 * 6,
) -> AllResults:
    return DFS(
        qc,
        circ_name=circ_name,
        metric=metric,
        pruning=[PruneColourCycle()],
        func_opt=func_opt,
        max_duration=max_duration,
    ).run()


def dfs_limit_colour_change(
    qc: QuantumCircuit,
    circ_name: str | None,
    metric: list = [MetricTcount()],
    func_opt: str | None = None,
    max_duration=60 * 60 * 6,
) -> AllResults:
    return DFS(
        qc,
        circ_name=circ_name,
        metric=metric,
        pruning=[PruneColourChange()],
        func_opt=func_opt,
        max_duration=max_duration,
    ).run()


def dfs_no_colour_cycle_kill_non_circuits(
    qc: QuantumCircuit,
    circ_name: str | None,
    metric: list = [MetricTcount()],
    func_opt: str | None = None,
    max_duration=60 * 60 * 6,
) -> AllResults:
    return DFS(
        qc,
        circ_name=circ_name,
        metric=metric,
        pruning=[PruneColourCycle(), PruneNonCircuits()],
        func_opt=func_opt,
        max_duration=max_duration,
    ).run()


def dfs_no_colour_cycle_limit_colour_change(
    qc: QuantumCircuit,
    circ_name: str | None,
    metric: list = [MetricTcount()],
    func_opt: str | None = None,
    max_duration=60 * 60 * 6,
) -> AllResults:
    return DFS(
        qc,
        circ_name=circ_name,
        metric=metric,
        pruning=[PruneColourCycle(), PruneColourChange()],
        func_opt=func_opt,
        max_duration=max_duration,
    ).run()


def dfs_max_depth(
    qc: QuantumCircuit,
    circ_name: str | None,
    metric: list = [MetricTcount()],
    func_opt: str | None = None,
    max_duration=60 * 60 * 6,
) -> AllResults:
    return DFS(
        qc,
        circ_name=circ_name,
        metric=metric,
        pruning=[PruneMaxDepth()],
        func_opt=func_opt,
        max_duration=max_duration,
    ).run()


def dfs_no_colour_cycle_max_depth(
    qc: QuantumCircuit,
    circ_name: str | None,
    metric: list = [MetricTcount()],
    func_opt: str | None = None,
    max_duration=60 * 60 * 6,
) -> AllResults:
    return DFS(
        qc,
        circ_name=circ_name,
        metric=metric,
        pruning=[PruneNonCircuits(), PruneMaxDepth()],
        func_opt=func_opt,
        max_duration=max_duration,
    ).run()


def dfs_strict_convergence(
    qc: QuantumCircuit,
    circ_name: str | None,
    metric: list = [MetricTcount()],
    func_opt: str | None = None,
    max_duration=60 * 60 * 6,
) -> AllResults:
    return DFS(
        qc,
        circ_name=circ_name,
        metric=metric,
        pruning=[PruneWorseCircuits()],
        func_opt=func_opt,
        max_duration=max_duration,
    ).run()
