% This is the proposed successive pseudoconvex approximation algorithm for
% SEE maximization (without QoS constraints)
clear;
clc;

K         = 7; % the number of users
M_tx   = 8; % the number of transmit antennas
M_rx   = 4; % the number of receive antennas

% power_dBm=36; power=M_tx*10^(power_dBm/10)/10^3;
power_dB              = 10; power = M_tx * 10^(power_dB / 10); % power constraint
sigma2_dB            = 0;   sigma2 = 10^(sigma2_dB / 10); % noise level
antenna_gain_dB = 16; antenna_gain = 10^(antenna_gain_dB / 10); % antenna gain

P0   = 1;  % static power consumption
rho  =  2.6; % power load coefficient
c_rd = 0.1; % coefficient of (quadratic) rate-dependent power consumption

MaxIter  = 50; % the maximum number of iterations
Sample = 100; % the number of repeatitions in Monte Carlo simulations

% the achieved SEE versus the CPU time
time_y = zeros(Sample, MaxIter+1); %"_y" denotes the proposed algorithm

% the achieved SEE versus the number of iterations
SumEE_y = zeros(Sample, MaxIter+1); % proposed method

for s = 1: 1: Sample
    disp(['Sample ' num2str(s)]);

    % generating the channel coefficients
    main_Channel
    loss_exponent = 2;
    for k = 1: 1: K
        for j = 1: 1: K
            H(:, :, k, j) = 1/sqrt(2) * (randn(M_rx, M_tx) + 1i * randn(M_rx, M_tx));
            H(:, :, k, j) = sqrt(antenna_gain) * H(:, :, k, j) / sqrt(distance(k, j) ^ loss_exponent);
        end
    end
    
    % the proposed successive pseudoconvex approximation algorithm
    % initial point
    for k = 1: 1: K
        Q(:, :, k) = eye(M_tx) / M_tx * power; % Q is the transmit covariance matrix
    end

    % initial noise and interference covariance matrix
    R = FUN_NI_cov(sigma2, H, Q, K, M_tx, M_rx);
    
    % initial INDIVIDUAL transmission rate and INDIVIDUAL energy efficiency
    [UserRate, UserEE, UserEnergy] = FUN_Rate_and_EE(K, R, Q, H, P0, rho, c_rd);
    
    % initial SUM energy efficiency
    SumEE_y(s, 1)   = sum(UserEE);
    SumRate_y(s,1) = sum(UserRate);

    for t = 1:1:MaxIter % this iterative algorithm is repeated for MaxIter times
        disp(['Proposed pseudoconvex algorithm: iteration ' num2str(t - 1) ',  Sum EE ' num2str(SumEE_y(s, t))]);        
        tic;
        
        %% calculate the gradient
        % Pi is the pricing matrix defined in (20b)
        [Pi, Gradient_Plus] = FUN_GradientMatrix_SEE(H, R, Q, K, UserEnergy); % Pi is defined in (20b)
        for k = 1:1:K
            % gradient of the approximate function's numerator w.r.t. Q_k
            numerator_gradient(:, :, k)   = Pi(:, :, k) + (Gradient_Plus(:, :, k, k));
            
            % gradient of the approximate function's denominator w.r.t. Q_k
            denominator_gradient(:, :, k) = rho * eye(M_tx) + 2 * c_rd * UserRate(k) * (Gradient_Plus(:, :, k, k));
            
            % gradient of SEE function w.r.t. Q_k
            SEE_gradient(:, :, k)         = numerator_gradient(:, :, k)/UserEnergy(k) - UserRate(k) * denominator_gradient(:, :, k)/UserEnergy(k)^2;
        end

        %% Dinkelbach's algorithm for the approximate problem (26)
        lambda = UserEE;
%         lambda = zeros(1, K);
        for tau = 1: 1: 200
            for k = 1: 1: K
                Q_lambda(:, :, k) = FUN_WaterFilling(H(:, :, k, k), R(:, :, k), Pi(:, :, k) - lambda(k) * denominator_gradient(:, :, k), power);
            end
            
            % to evaluate numerator and denominator function
            numerator_dinkelbach   = FUN_Rate_and_EE(K, R, Q_lambda, H, P0, rho, c_rd);
            denominator_dinkelbach = UserEnergy;
            for k = 1: 1: K
                numerator_dinkelbach(k)     = numerator_dinkelbach(k) + real(trace(Pi(:, :, k)' * (Q_lambda(:, :, k) - Q(:, :, k))));
                denominator_dinkelbach(k) = denominator_dinkelbach(k) + real(trace(denominator_gradient(:, :, k)' * (Q_lambda(:, :, k) - Q(:, :, k))));
                lambda_new(k)                      = numerator_dinkelbach(k) / denominator_dinkelbach(k);
            end
            % disp(['The precision of Dinkelbach Algorithm is ' num2str([norm(lambda_new-lambda)])]);
            if abs(norm(lambda_new - lambda)) <= 10^-5
                BQ = Q_lambda;
                break;
            else
                lambda = lambda_new;
            end
        end
        
        %% to calculate the stepsize by the successive line search ("ascent" is the product between update direction and gradient)
        alpha = 0.01; %parameters in successive line search
        beta  = 0.8; %parameters in successive line search
        
        for k = 1: 1:K
            ascent(k) = real(trace((BQ(:, :, k) - Q(:, :, k))' * SEE_gradient(:, :, k)));
        end

        stepsize = 1;
        while 1
            Q_tmp = Q + stepsize * (BQ - Q);
            R_tmp = FUN_NI_cov(sigma2, H, Q_tmp, K, M_tx, M_rx); % noise and interference covariance matrix
            
            [UserRate_tmp, UserEE_tmp, UserEnergy_tmp] = FUN_Rate_and_EE(K, R_tmp, Q_tmp, H, P0, rho, c_rd);
            
            objval_tmp = sum(UserEE_tmp);
            if objval_tmp >= SumEE_y(s, t)+alpha * stepsize * sum(ascent), 
                SumEE_y(s, t+1) = objval_tmp;
                Q               = Q_tmp;
                R               = R_tmp;
                UserRate = UserRate_tmp;
                SumRate_y(s,t+1)= sum(UserRate);
                UserEE    = UserEE_tmp;
                UserEnergy = UserEnergy_tmp;
                break;
            else
                stepsize = stepsize * beta;
            end
        end
        %%
        time_y(s, t+1) = time_y(s, t) + toc;
    end
end