% This is the proposed successive pseudoconvex approximation algorithm for
% GEE maximization (without QoS constraints)
clear;
clc;

K       = 7; % the number of users
M_tx = 8; % the number of transmit antennas
M_rx = 4; % the number of receive antennas

power_dB             = 10;  power  = M_tx * 10 ^ (power_dB / 10); % power constraint
sigma2_dB           = 0;    sigma2 = 10 ^ (sigma2_dB/10); % noise level
antenna_gain_dB = 16; antenna_gain = 10^(antenna_gain_dB/10); % antenna gain

P0  = 16; 
rho = 2.6; %power load coefficient
c    = 0.1; %coefficient of (qudrattic) rate-dependent power consumption

MaxIter  = 50; % the maximum number of iterations
Sample = 10; % the number of Monte-Carlo repeatitions

% the achieved GEE versus the number of iterations
GEE_y = zeros(Sample, MaxIter + 1); %"_y" denotes the proposed algorithm

% the achieved sum rate versus the number of iterations
SumRate_y = zeros(Sample, MaxIter + 1);

% the achieved SEE versus the CPU time
time_y = zeros(Sample, MaxIter + 1);

%parameters in successive line search
alpha = 0.01;
beta   = 0.5;

for s = 1: 1: Sample,
    disp(['Sample ' num2str(s)]);

    % generating channel coefficients
    main_Channel % this file is used to generate the positions of users and calculate their distance, cf. Figure 1
    loss_exponent = 2;
    for k = 1: 1: K,
        for j = 1: 1: K,
            H(:, :, k, j) = 1/sqrt(2) * (randn(M_rx, M_tx) + 1i * randn(M_rx, M_tx));
            H(:, :, k, j) = sqrt(antenna_gain) * H(:, :, k, j) / sqrt(distance(k, j) ^ loss_exponent); % "distance" is generated by "main_channel"
        end;
    end;
    
    
    % the proposed successive pseudoconvex approximation algorithm
    % initial point
    for k = 1: 1: K,
        Q(:,:,k) = eye(M_tx) / M_tx * power; % Q is the transmit covariance matrix
    end;
    R = FUN_NI_cov(sigma2,H,Q,K,M_tx,M_rx); % initial noise and interference covariance matrix
    [SumRate_y(s,1), rate_plus, rate_minus] = FUN_SumRate(K,R,Q,H); % initial sum rate

    SumPower = 0; % initial total power consumption
    for k = 1: 1: K
        SumPower = SumPower + P0 + rho * trace(Q(:,:,k)) + c * (rate_plus(k) - rate_minus(k)) ^ 2;
    end
    GEE_y(s,1) = SumRate_y(s,1) / SumPower; % initial GEE
    
    for t = 1: 1: MaxIter, % this iterative algorithm is repeated for MaxIter times
        disp(['the proposed successive pseudoconvex approximation algorithm: iteration ' num2str(t-1) ', achieved GEE ' num2str(GEE_y(s,t))]);
        
        tic;
        
        % noise and interference covariance matrix
        R = FUN_NI_cov(sigma2,H,Q,K,M_tx,M_rx);
        
        % Pi(k): the sum of the gradient of "other" rate functions w.r.t. Q_k, see (7)
        % the individual transmission rate function is written as the difference of two functions (_plus and _minus), as in the equation before (34)
        [SumRate, rate_plus, rate_minus] = FUN_SumRate(K,R,Q,H);
        numerator_value = SumRate;
        denominator_value = 0;
        for k = 1: 1: K
            denominator_value = denominator_value + P0 + rho * trace(Q(:,:,k)) + c * (rate_plus(k) - rate_minus(k)) ^ 2;
        end
        
        [Pi, Gradient_Plus, Gradient_Minus] = FUN_GradientMatrix(H, R, Q, K);
        
        for k = 1: 1: K,
            % gradient of the numerator (sum rate function) w.r.t. Q_k
            numerator_gradient(:, :, k) = Pi(:, :, k) + (Gradient_Plus(:, :, k, k) - Gradient_Minus(:, :, k, k)); 
            
            % gradient of the denominator w.r.t. Q_k
            denominator_gradient(:, :, k) = rho * eye(M_tx) + 2 * c * (rate_plus(k) - rate_minus(k)) * (Gradient_Plus(:, :, k, k) - Gradient_Minus(:, :, k, k));

            % gradient of GEE function w.r.t. Q_k
            GEE_gradient(:, :, k) = numerator_gradient(:, :, k) / denominator_value - numerator_value * denominator_gradient(:, :, k) / denominator_value ^ 2;
        end;
        
        % Dinkelbach's algorithm for the approximate problem
        lambda = numerator_value / denominator_value;
%         lambda=0;
        for tau = 1: 1: 200,
            for k = 1: 1: K,
                Q_lambda(:, :, k) = FUN_WaterFilling(H(:, :, k, k), R(:, :, k), Pi(:, :, k) - lambda * denominator_gradient(:, :, k), power);
            end;
            
%             to evaluate numerator and denominator function
            numerator_dinkelbach = FUN_SumRate(K, R, Q_lambda, H);
            denominator_dinkelbach = denominator_value;
            for k = 1: 1: K,
                numerator_dinkelbach = numerator_dinkelbach + real(trace(Pi(:,:,k)' * (Q_lambda(:,:,k) - Q(:,:,k))));
                denominator_dinkelbach = denominator_dinkelbach + real(trace(denominator_gradient(:,:,k)' * (Q_lambda(:,:,k) - Q(:,:,k))));
            end;
            
            % to update lambda
            lambda_new = numerator_dinkelbach / denominator_dinkelbach;
            %         disp(['The precision of Dinkelbach Algorithm is ' num2str([lambda lambda_new abs(lambda_new-lambda)])]);
            if abs(lambda_new - lambda) <= 10^-5,
                BQ = Q_lambda;
                break;
            else
                lambda = lambda_new;
            end;
        end;
        
%         to calculate the stepsize by the successive line search ("ascent" is the product between update direction and gradient)
        for k = 1: 1: K,
            ascent(k) = real(trace((BQ(:, :, k) - Q(:, :, k))' * GEE_gradient(:, :, k)));
        end;
%         [ascent sum(ascent)]
        
        stepsize = 1;
        while 1,
            Q_tmp = Q + stepsize * (BQ - Q);
            R_tmp = FUN_NI_cov(sigma2, H, Q_tmp, K, M_tx, M_rx); % noise and interference covariance matrix
            [SumRate_tmp, rate_plus_tmp, rate_minus_tmp] = FUN_SumRate(K, R_tmp, Q_tmp, H);
            SumPower_tmp = 0;
            for k = 1: 1: K,
                SumPower_tmp = SumPower_tmp + P0 + rho * trace(Q_tmp(:,:,k)) + c * (rate_plus_tmp(k) - rate_minus_tmp(k))^2;
            end;

            gee_tmp = SumRate_tmp / SumPower_tmp;
            if gee_tmp >= GEE_y(s,t) + alpha * stepsize * sum(ascent),
                SumRate_y(s, t + 1) = SumRate_tmp;
                GEE_y(s, t + 1) = gee_tmp;
                Q = Q_tmp;
                break;
            else
                stepsize = stepsize * beta;
            end;
        end;
        time_y(s, t + 1) = time_y(s, t) + toc;
        
% %         to test if the GEE function is increased
%         gee_y(s,t+1)>gee_y(s,t)
    end;
end;